#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>

#define F_CPU 20000000
#include <util/delay.h>


// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
// look ddsControl, dds inside ISR1
// LCD connection check !


// Use compiler option -O1

// port b0 pin14 = clipled

// PORTD.4 pin 6 show ISR1 timing
// PORTD.3 pin 5 show doIQsample timing
// PORTB.7 pin 10 

// D.3 is also OC2B
// use here for f output
#define IO2pin 3
#define IO2port PORTD

#define IO3pin 2
#define IO3port PORTB

#define IO4pin 7
#define IO4port PORTB


// Frontend parameters
//
// Md   number of input-sequence samples used until one I/Q-integration output 
//      value is generated by the ISR and put into the FIFO
//      values go at half-rate into downsampling two-stage CIC filters
//
// Ns   number of clock cycles between ADC sampling
//
// Mc   order of the smoothing CIC filter behind the downsampling
//
// P    the ouput of doIQsample routine from the FIFO is divided by P before
//      it enters the second CIC-Filter. This is used to prevent overflow
//      in the second CIC filter and in the CORDIC subroutine.
//
//      for scaling of frontend for correct uMean computation:
//      use :     (Mc/P) * (Md^2/2^2) == 32
//      for fast division P should be a power of 2
//

//#define RTTY125
#define RTTY147

#ifdef RTTY147
  #define Md  16
  #define Ns 2559UL
  #define Mc 4 
  #define OCRoffset 1205
  #define P 8    
  #define FRQlength 8
  #define HEADLINE "RTTY_147.3"
#endif




//----------------------------------------------------------------------------------------


//----------------------------------------------------------------------------------------  


void delay_ms(uint16_t x){
  uint16_t k ;
  for (k=0 ; k<x ; k++ ) { _delay_loop_2(4000) ; }
  }



//----------------------------------------------------------------------------------------  


//----------------------------------------------------------------------------------------
//
// RS232 TX routines

// define length of FIFO: UARTMASK+1 , UARTMASK must be of form 2^n-1 (all binary ones)
//
#define UARTMASK 255

volatile uint8_t uartFIFO[256] ;
// for atomic read/write the following must be 1-Byte variables
volatile uint8_t uartPUTidx ;
volatile uint8_t uartGETidx ;

void uartFIFOinit(){
// uartPutidx==uartGETidx signals FIFO is empty
  uartPUTidx=0 ;
  uartGETidx=0 ;
  }

void uartPutc(char c){
// put char into FIFO and output later via doIQsample()
  uartFIFO[uartPUTidx]=c ;
// the following write to uartPUTidx is atomic, so even an interrupt in
// between may happen without fault
  uartPUTidx=(uartPUTidx+1) & UARTMASK ;
  }

// direct output
int uartSnd(unsigned char c){
   while (!(UCSR0A & (1<<UDRE0))); 
   UDR0 = c;                      
   return 0;
   }

void uartPutsPgm(const char *progmem_s){
  register char c;
  while ( (c = pgm_read_byte(progmem_s++)) ) { 
    uartPutc(c); 
	}
  }

void uartBlank(){
  uartPutc(' ') ; 
  }

void uartCrlf(){
  uartPutc(13) ; 
  uartPutc(10) ;
  }

void uartLf(){
  uartPutc(13) ; 
  }

void uartHex(uint8_t v){
  v &=0b00001111 ;
  if (v<=9) { uartPutc('0'+v) ; } 
       else { uartPutc(v+'A'-10) ; } ;
  }

void uartByte(uint8_t v){
  uartHex(v>>4) ; 
  uartHex(v) ; 
  }

void uartWord(uint16_t v){
  uartByte(v>>8) ; 
  uartByte(v) ; 
  }

void uartUint32(uint32_t v){
  uartWord(v>>16) ; 
  uartWord(v) ; 
  }

void uartDecOut(int k){
   uartPutc( ((k/10000) % 10)+48 ) ;
   uartPutc( ((k/1000) % 10)+48 ) ;
   uartPutc( ((k/100) % 10)+48 ) ;
   uartPutc( ((k/10) % 10)+48 ) ;
   uartPutc( (k % 10)+48 ) ;
   uartPutc(32) ;
   }

void uartDecOut32(uint32_t k){
   uartPutc( ((k/100000000UL) % 10)+48 ) ;
   uartPutc( ((k/10000000UL) % 10)+48 ) ;
   uartPutc( ((k/1000000UL) % 10)+48 ) ;
   uartPutc( ((k/100000UL) % 10)+48 ) ;
   uartPutc( ((k/10000) % 10)+48 ) ;
   uartPutc( ((k/1000) % 10)+48 ) ;
   uartPutc( ((k/100) % 10)+48 ) ;
   uartPutc( ((k/10) % 10)+48 ) ;
   uartPutc( (k % 10)+48 ) ;
   uartPutc(32) ;
   }
//----------------------------------------------------------------------------------------  
// uart is called with 500 samples/sec interrupt driven
// DDH baud rate is 50bd, means 1 bit=10... samples
//----------------------------------------------------------------------------------------  
//
// software uart for BAUDOT code
//

#define LFcode 10
#define CRcode 13
#define UU     'y'

// 'x' = who-is-there
// 'y' = unused
// 130=ltrs 131=figs 


prog_int8_t BaudotTable []={
  'o'    , 'E', LFcode , 'A', ' ', 'S', 'I', 'U',
  CRcode , 'D', 'R'    , 'J', 'N', 'F', 'C', 'K',
  'T'    , 'Z', 'L'    , 'W', 'H', 'Y', 'P', 'Q',
  'O'    , 'B', 'G'    , 130, 'M', 'X', 'V', 131, 
  'o'    , '3', LFcode , '1', ' ', '"', '8', '7',
  CRcode , 'x', '4'    , 'b', ',',  UU, ':', '(',
  '5'    , '+', ')'    , '2',  UU, '6', '0', '1',
  '9'    , '?', '&'    , 130, '.', '/', '=', 131 } ;

uint8_t uartTimer ;
uint8_t uartShiftReg ;
uint8_t uartBitCount ;
uint8_t baudotShift ;

void BaudotPrint(uint8_t baudotChar){
   uint8_t AsciiChar ;
   AsciiChar= pgm_read_byte( BaudotTable +  baudotChar+(baudotShift<<5)  ) ;
   if ( AsciiChar==131) { 
     baudotShift=0 ; 
	 AsciiChar=0 ; 
	 }
    else if ( AsciiChar==130 ) {
	 baudotShift=1 ; 
	 AsciiChar=0 ; 
	 }
   if (AsciiChar!=0) { 
     // display only printable characters
     uartPutc(AsciiChar) ; 
	 }
  }




void softUartInit(){
  uartShiftReg=0 ;
  uartTimer=0 ;
  baudotShift=0 ;
  }


uint8_t uartStart ; // only used for display purposes

#define uartFullTime 10
#define uartHalfTime 6

void checkForStartBit(uint8_t input){
  if ( input != 0) { 
    uartTimer=uartHalfTime ;
    uartBitCount=0 ; 
    uartShiftReg=0 ; 
    uartStart=1 ; 
    }
  }

void uartSample(uint8_t input){
  if (uartTimer==0) {
   	checkForStartBit(input) ;
    }
   else { 
    uartTimer++ ;
	}
  
  if (uartTimer>uartFullTime) {
    // a full bit time has elapsed again
    uartTimer -= uartFullTime ;
    // shift input into shiftregister
    if ( input == 0) { 
	  uartShiftReg=uartShiftReg+(1 << uartBitCount) ; 
	  }
    // count bits
    uartBitCount++ ;
    if (uartBitCount==6) {
	  // we have enough bits
      uartShiftReg=uartShiftReg >>1  ; // eliminate stop-bit
      BaudotPrint(uartShiftReg) ;
      }
    if (uartBitCount>6) {
	  // too many bits, restart 
	  uartBitCount=0 ; 
	  uartTimer=0 ; 
	  }
    }
  }





//----------------------------------------------------------------------------------------  
//----------------------------------------------------------------------------------------  

// generated by scaling with 65536.0 :: CORDIC1.PAS
int32_t iCordicSinTab[5]={25080L,12785L,6424L,3216L,1608L} ;
int32_t iCordicCosTab[5]={60547L,64277L,65220L,65457L,65516L} ;


int16_t Amplitude,LogAmplitude ;

uint8_t iCordic(int32_t x, int32_t y) {
// compute angle and radius of point (x,y)
// angle is valid from 0 to 255 (256 would mean 2*Pi )
// angle is stored in phi and returned via function
// radius is stored in Amplitude and mus be lower than 2^15-1
// warning, if x near 2^15 and y near 2^15 than radius is too big !

  int32_t s,c,x0,y0,x1,y1,tmp ;
  uint8_t sign,phi,phi1,phiTest,k_cordic ;

// first rotate until x0,y0 are in octant 0    
// 
  x0=x ; y0=y ;
  phi=0 ;
  if ( y0<0 ) { x0=-x0 ; y0=-y0 ; phi=128 ; }
  if ( x0<0 ) { tmp=x0 ; x0=y0 ; y0=-tmp ; phi +=  64 ; }
  sign=0 ;
  if (y0>x0) { sign=1 ; phi += 64 ; tmp=x0 ; x0=y0 ; y0=tmp ; }
  // now x0 >= y0 >=0
 
 // warning: this scales amplitude !! so leave out
 // while (x0>0x7fff) { x0=x0 >> 1 ; y0=y0 >> 1 ; }
 // now rotate in 4 SAR steps with phiTest=16,8,4,2,1
 // and check that x0,y0 stays in octant 0

  phiTest=16 ;
  phi1=0 ;
  k_cordic=0 ;
  while (k_cordic<=4){
    // get cosine(phiTest)=c and sine(phiTest)=ss from table
    c=iCordicCosTab[k_cordic] ;
    s=iCordicSinTab[k_cordic] ;
    // rotate (x0,y0) by phi into (x1,y1)
    x1=( c*x0+s*y0) / 65536L ;
    y1=(-s*x0+c*y0) / 65536L ;
    // change x0,y0 if valid rotation step meaning that we stay in octant 0
    if (y1>=0) { x0=x1 ; y0=y1 ; phi1 += phiTest; }
    phiTest = phiTest >> 1 ;
	k_cordic++ ;
    }
  Amplitude=x0 ;
  if (sign) { phi -= phi1 ; } else { phi += phi1 ; }
  return phi ;
  }




//----------------------------------------------------------------------------------------  
//
//----------------------------------------------------------------------------------------  
//

uint8_t LogTab[16]={ 0,9,17,25,32,39,46,52,58,64,70,75,81,86,91,95 } ;


uint16_t Log2(uint32_t x){
// compute 100*ld(x) , so Log2(1024) = 1000
  uint16_t v ;
  v=3100 ;
  // shift until a bit in MSB-byte is set, each byte shift reults in log += 800
  while ( (x&0xff000000L )==0 ) { x=x<<8 ; v -= 800 ; }
  // shift until most significant bit is set, each bit-shift results in log += 100
  while ( (x&0x80000000L )==0 ) { x=x<<1 ; v -= 100 ; }
  // x has now form 1iii ixxx xxxx xxxx xxxx xxxx xxxx xxxx
  // get the next 4 bits =iiii and address table with it
  uint8_t i=(x>>27) & 0xf ;
  v += LogTab[i] ;
  return v ;

}


//----------------------------------------------------------------------------------------

uint8_t Clip ;
uint8_t ClipCount ;

#define ClipLed 0

void doClipCheck(){
  // CLIP-FLAG is set by DC routine doIQsample() if input out of range

  if (Clip) { PORTB |= _BV(ClipLed) ; } else { PORTB &= ~ _BV(ClipLed) ;}
  ClipCount++ ;
  // reset CLIP-FLAG all 256 steps
  if (ClipCount==0){ Clip=0 ; }
  }


//----------------------------------------------------------------------------------------
// signal stuff

uint16_t sampleTime ;
int16_t ADCv ;
volatile int8_t DAC1,DAC2 ;


uint8_t CordicPhase ;
uint8_t LastCordicPhase ;
int8_t  PhaseDifference ;
int16_t ExtendedPhase ;


int32_t uMean ;
int16_t meanCount ;


//----------------------------------------------------------------------------------------

typedef int32_t CICtype  ;

volatile CICtype IIintegrator1 ;
volatile CICtype QQintegrator1 ;

volatile CICtype IIintegrator2 ;
volatile CICtype QQintegrator2 ;

volatile CICtype  IIsample ;
volatile CICtype  QQsample ;

#define IntFifoSize 8
#define IntFifoMask 0x07

volatile uint8_t IntFifoInPtr ;
volatile uint8_t IntFifoOutPtr ;
volatile CICtype  IntFifoII[IntFifoSize] ;
volatile CICtype  IntFifoQQ[IntFifoSize] ;

//----------------------------------------------------------------------------------------
// for 500Hz sample generation
volatile uint8_t ClkMail500 ;
#define TTT 40000
uint16_t Ticks500 ;

void timer1Init(){

  TCCR1A=0b10000010 ; // fast pwm
  TCCR1B=0b00011001	; // ck=XTAL
  // Interupt enable for Timer1-Overflow
  TIMSK1=0b00000001 ;
  OCR1A=OCRoffset ;
  ICR1=Ns-1 ; // sample period predivider


  IntFifoInPtr=0 ;
  IntFifoOutPtr=0 ;
  
  IIintegrator1=0 ;
  QQintegrator1=0 ;
  IIintegrator2=0 ;
 
  QQintegrator2=0 ;

  sampleTime=0 ;
  CordicPhase=0 ;
  LastCordicPhase=0 ;
  ExtendedPhase=0 ;

  }


ISR(TIMER1_OVF_vect) {
   IO3port |=  _BV(IO3pin) ;
// get sample from input and convert to signed value
  ADCv=( (int16_t)ADC-512) ; 
// clip ADC value in order to detect ADC overflow
#define ADCMAX 500 
  if (ADCv > ADCMAX) { ADCv=ADCMAX ; Clip=1 ; }
  if (ADCv <-ADCMAX) { ADCv=-ADCMAX ;  Clip=1 ; }
// ADCv now is in -ADCMAX..ADCMAX

// multiply and sum samples with sequence 1  0  -1  0  1  0  -1  0  1  0 ... for I channel
// multiply and sum samples with sequence 0  1  0  -1  0  1  0  -1  0  1 ... for Q channel
// the summing directly is done as two stage intergration of a CIC filter.
// The CIC integration outputs are in IcicInt2 and QcicInt2

  if (sampleTime & 1) {
    if (sampleTime & 0b10) { IIintegrator1 += ADCv ; } else { IIintegrator1 -= ADCv ; }
	IIintegrator2 += IIintegrator1 ;
    }
   else {
    if (sampleTime & 0b10) { QQintegrator1 += ADCv ; } else { QQintegrator1 -= ADCv ; }
	QQintegrator2 += QQintegrator1 ; 
	}

  sampleTime++ ;

// serve UART from uartFIFO in between if data is in FIFO and uart-reg empty
  if ( (uartPUTidx != uartGETidx) & ((UCSR0A & (1<<UDRE0))!=0) ) {
    uint8_t c=uartFIFO[uartGETidx] ;
	uartGETidx=(uartGETidx+1) & UARTMASK ;
	UDR0 = c;  
    }
  
  OCR2A = DAC1 ;
  OCR2B = DAC2 ;

// after Md samples output I/Q values to FIFO 

  if (sampleTime==Md) {
    sampleTime=0 ;
// put CIC integration values downsampled by Md into FIFOs
    IntFifoII[IntFifoInPtr]=IIintegrator2 ;
	IntFifoQQ[IntFifoInPtr]=QQintegrator2 ;
	IntFifoInPtr=(IntFifoInPtr+1) & IntFifoMask ;
    }

  PORTD &= ~ _BV(2) ;  
  Ticks500 += Ns ;
  if(Ticks500>=TTT) {
 	ClkMail500=1 ;  
	Ticks500 -= TTT ;
	PORTD |= _BV(2) ;
    } 


  IO3port &= ~ _BV(IO3pin) ;
  }

//----------------------------------------------------------------------------------------

CICtype IIcomb1store,IIcomb1out ;
CICtype IIcomb2store,IIcomb2out ;
CICtype QQcomb1store,QQcomb1out ;
CICtype QQcomb2store,QQcomb2out ;



int32_t IIfifo[Mc] ;
int32_t IIintegrator3 ;
int32_t IIcic3out ;

int32_t QQfifo[Mc] ;
int32_t QQintegrator3 ;
int32_t QQcic3out ;

uint8_t CICfifoPTR ;

void CICfifoINIT(){
  for ( CICfifoPTR=0 ; CICfifoPTR<Mc ; CICfifoPTR++){
    QQfifo[CICfifoPTR]=0 ;
    IIfifo[CICfifoPTR]=0 ;
    }
  CICfifoPTR=0 ;
  IIintegrator3=0 ;
  QQintegrator3=0 ;
  }


int32_t FRQfifo[FRQlength] ;
int32_t FRQintegrator ;
int32_t FRQcicout ;
uint8_t FRQfifoPTR ;


int8_t PhaseDifference ;

int8_t BitLine ;

void FRQdemodInit(){
  for ( FRQfifoPTR=0 ; FRQfifoPTR<FRQlength ; FRQfifoPTR++){
    FRQfifo[FRQfifoPTR]=0 ;
    }
  FRQfifoPTR=0 ;
  FRQintegrator=0 ;
  }






// doIQsample() fetches data from the I and Q FIFO
// these data are feed to the 2 stage COMB section of the CIC downsampling filter
// Phase and amplitude are then computed via a CORDIC-like rotation method

void doIQsample(){
// signalize timing
 

// fetch data from FIFO
  IIsample=IntFifoII[IntFifoOutPtr] ;
  QQsample=IntFifoQQ[IntFifoOutPtr] ;
  IntFifoOutPtr=(IntFifoOutPtr+1) & IntFifoMask ;

// do 1.st comb-step of two stage downsampling CIC filter
  IIcomb1out=IIcomb1store-IIsample ; IIcomb1store=IIsample ; 
  QQcomb1out=QQcomb1store-QQsample ; QQcomb1store=QQsample ; 


// do 2.nd comb-step of two stage downampling CIC filter
  IIcomb2out=IIcomb2store-IIcomb1out ; IIcomb2store=IIcomb1out ; 
  QQcomb2out=QQcomb2store-QQcomb1out ; QQcomb2store=QQcomb1out ; 

 

// IcombOutb and QcombOutb are now the downsampled filtered I and Q channel values.
// these go now to a further 1-stage smoothing CIC filter of order Mc,
// but now further downsampling occurs

// integration step of smoothing CIC filter I-channel
   IIintegrator3 += IIcomb2out / P ;
// comb staep of smoothing CIC filter
   IIcic3out = IIintegrator3 - IIfifo[CICfifoPTR] ;
// FIFO update of smoothing CIC filter
  IIfifo[CICfifoPTR] =  IIintegrator3 ;

// same for I-channel
  QQintegrator3 += QQcomb2out / P ;
  QQcic3out = QQintegrator3 - QQfifo[CICfifoPTR] ;
  QQfifo[CICfifoPTR] =  QQintegrator3 ;
// advance pointer and bump around
  CICfifoPTR++ ; if ( CICfifoPTR==Mc ) { CICfifoPTR=0 ; }

// CICfifoIout and CICfifoQout are now the triple filtered I and Q channel values
// these are used to compute phase and amplitude using a CORDIC maethod


 // IO2port |=  _BV(IO2pin) ;
  CordicPhase=iCordic( IIcic3out , QQcic3out ) ;
 // IO2port &= ~ _BV(IO2pin) ;



  LogAmplitude=Log2(Amplitude+1) ; 

//  DAC1 = 128+CICfifoIout/128 ;
//  DAC2 = 128+CICfifoQout/128 ;

 


  int16_t x1=(LogAmplitude-390)/4 ;
  if (x1<0) { x1=0 ; }
  if (x1>255) { x1=255 ; }
//  DAC1= x1 ;
//DAC1= 128+CordicPhase ;

  PhaseDifference=CordicPhase-LastCordicPhase ;
  LastCordicPhase=CordicPhase ;
  ExtendedPhase += PhaseDifference ;

 
  FRQintegrator += PhaseDifference ;
  FRQcicout = FRQintegrator - FRQfifo[FRQfifoPTR] ;
  FRQfifo[FRQfifoPTR] =  FRQintegrator ;
// advance pointer and bump around
  FRQfifoPTR++ ; if ( FRQfifoPTR==FRQlength ) { FRQfifoPTR=0 ; }
// normalize
  FRQcicout= FRQcicout / FRQlength ;

  if (uartStart) { DAC2=60 ; uartStart=0 ; } else { DAC2=uartTimer*4 ; }
  DAC1=128  + FRQcicout  ;
  }





//----------------------------------------------------------------------------------------
/*
volatile uint8_t ClkMail ;
uint8_t Clk ;

void timer2Init(){
  // Timer 2 BitClk
  TCCR2A=0b00000010 ; // ctc mode 2
  //  TCCR2B=0b00000111 ; // prescale by 1024
  //  TCCR2B=0b00000110 ; // prescale by 256
  TCCR2B=0b00000100 ; // prescale by 64
  // Parameter for 20000000Hz XTAL
  OCR2A=125-1 ; //  top
  OCR2B=100 ; //  compare
// Interupt enable for Timer2-compare-B
  TIMSK2=0b00000100 ;
  Clk=0 ;
  ClkMail=0 ;
  }

ISR(TIMER2_COMPB_vect) {  
  Clk++ ;
  if (Clk==5) { Clk=0 ; ClkMail=1 ; } 
  }
*/

//volatile uint8_t ClkMail ;



void timer2Init(){
  // Timer 2 BitClk
  TCCR2A=0b10100011 ; // fast pwm top=0xFF OC2A=pin 17/28 OC2B=pin 5/28
  //TCCR2B=0b00000010 ; // prescale by 8 
   TCCR2B=0b00000001 ; // prescale by 1 
  OCR2A=50; 
  OCR2B=150 ; 
// Interupt enable for Timer2-overflow
 // TIMSK2=0b00000001 ; // TOIE2

  ClkMail500=0 ;
 
  }

// timer 2 rate : 20MHz/8/256=9765.625 interrupts/sec
/*
uint16_t T2ticks ;

ISR(TIMER2_OVF_vect) {
 // PORTD &= ~ _BV(2) ;  
  T2ticks+=256 ;
  if(T2ticks>40000U) {
 	ClkMail=1 ;  
	T2ticks -= 40000U ;
//	PORTD |= _BV(2) ;
    } 
  }
*/


//----------------------------------------------------------------------------------------  



void FMdemod(){
  int32_t frqEstimate ;
// get frequency in atomic way
  cli() ;
  frqEstimate=FRQcicout ;
  sei() ;

 
  if (frqEstimate >  5) { BitLine=0 ; }
  if (frqEstimate < -5) { BitLine=1 ; } 

  #ifdef RTTY125
  BitLine=BitLine ^ 1 ;
  #endif

 
//  DAC2=128  +(FMcicout) ;
//  if (uartStart) { DAC1=60 ; uartStart=0 ; } else { DAC1=uartTimer*4 ; }
 // DAC2=16*BitLine ;

  uartSample(BitLine) ;
  }

//----------------------------------------------------------------------------------------

void timer0Init(){
// 147.3 kHz
  TCCR0A=0b00100011 ; // fast pwm  output OCR0B OC0B=PD5 pin 11of28
  TCCR0B=0b00001001	; // ck=XTAL
  OCR0B=68 ;
  OCR0A=134-1 ; // sample period predivider 


  }


//----------------------------------------------------------------------------------


/*
void timer0Init(){
#define T0divider 20  
  TCCR0A=0b10100011 ; // mode 3 , fast pwm TOP=255 out OC0A and OC0B
  TCCR0B=0b00000001	; // ck=XTAL , 78.125kHz
  OCR0A = 50 ;
  OCR0B = 100 ;
  }
*/

//----------------------------------------------------------------------------------------



//----------------------------------------------------------------------------------------  



int main(){

  DDRD =0b11111111 ; // enable outputs , bit 0,1=input
  PORTD=0b11111111 ; // pullups on on input pins
  
  DDRB =0b11111111 ; // enable outputs 
  PORTB=0b11111111 ; // pullups on on input pins
  
  DDRC =0b11111110 ; // enable outputs , bit 0=input
  PORTC=0b11111110 ; // pullups off on ADC pins

//-------------------------------------------

  // ADMUX =0b01000000 ; // Vref=AVcc
  ADMUX =0b11000000 ; // Vref=1.1V

  ADCSRA=0b10100111 ; // 20MHz/128=156.25kHz    /13=12.019kHz = max sample frequency
  ADCSRB=0b00000110 ; // timer/counter1 overflow is trigger

 
// 20.00MHz/19200/16=65
// use 19200 bit/sec UBRRL=64
  UCSR0B |= ( 1 << TXEN0 );					// UART TX einschalten
  // UCSR0C |= ( 1 << URSEL )|( 3<<UCSZ0 );	    // Asynchron 8N1
  UBRR0H  = 0;                               // Highbyte ist 0
  UBRR0L  = 65-1 ;                             // 19200 Bit/sec

  uartFIFOinit() ;
  timer0Init() ;
  timer1Init() ;
  timer2Init() ;



  CICfifoINIT() ;
  FRQdemodInit() ;


  uartCrlf() ;
  uartPutsPgm(PSTR(HEADLINE)) ;
  uartCrlf() ;

 

  sei() ;
  while (1){ 
    if ( IntFifoInPtr!=IntFifoOutPtr ) { 
      doIQsample() ; 
	  doClipCheck() ;
	  }
    if ( ClkMail500 ) { 
      ClkMail500=0 ; 
      FMdemod() ;
      } ;
    }
  return(0) ;
  }
